/*
* Creation date : Tues Mar 29 09:00:00 2007
* Last modified : %modify_time%
*/
/** @file
* \brief This file contains implementation of 
* CMLA functions. 
*
* \version CE2_CMLA.c#1:csrc:1
* \author Yermalayeu Ihar
* \remarks Copyright (C) 2007 by Discretix Technologies Ltd.
* All Rights reserved
*/

/************************ Include Files ***********************/

#include "CE2_CMLA.h"
#include "LLF_CMLA.h"

/************************ Defines *****************************/
/************************ Enums *******************************/
/************************ Typedefs ****************************/
/************************ Global Data *************************/
/************************ Private function prototype **********/
/************************ Private Functions *******************/
/************************ Public Functions ********************/

/**
****************************************************************
* Function Name: 
*  CE2_CMLA_KDF
*
*  @param[in]  X        - Seed value, an octet string of length 128.
*  @param[out] KEK      - Key Encryption Key, an octet string of length 16.
*  
* @returns \b
*  CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - error code:
*   - CE2_CMLA_IS_NOT_SUPPORTED
*   - CE2_CMLA_KDF_INVALID_SEED_POINTER_ERROR
*   - CE2_CMLA_KDF_INVALID_KEK_POINTER_ERROR
*
* \brief \b 
* Description:
*  CMLA Key Derivation Function produce the Key Encrypting Key [CMLA v1.0-051221].
*  CE2_CMLA_KDF operates on an octet string of length 128.
*  The input splits in two blocks of equal length. A constant value is concatenated
*  to the first block. Then SHA-1 is applied to the preceding result. The result
*  and two integers derived from the second block are used in a modulus calculation
*  to produce the Key Encrypting Key. The output shall be 16 bytes.
*
*  \b 
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function LLF_CMLA_KDF. 
***************************************************************/
CE2CIMPORT_C  CE2Error_t CE2_CMLA_KDF(CE2_CMLA_SEED_t      X ,       /*in*/
                                   CE2_CMLA_KEK_t       KEK      /*out*/)
{
  if (X == DX_NULL)
    return CE2_CMLA_KDF_INVALID_SEED_POINTER_ERROR;

  if (KEK == DX_NULL)
    return CE2_CMLA_KDF_INVALID_KEK_POINTER_ERROR;

  return LLF_CMLA_KDF(X, KEK);
} /* End of CE2_CMLA_KDF */

/**
****************************************************************
* Function Name: 
*  CE2_CMLA_Wrap
*
*  @param[in]  KEK      - Key Encryption Key, an octet string of length 16 bytes.
*  @param[in]  Key      - Plain key, an octet string of length 32 bytes.
*  @param[out] WrapKey  - Wrapped Key, an octet string of length 40 bytes.
*  
* @returns \b
*  CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - error code:
*   - CE2_CMLA_IS_NOT_SUPPORTED
*   - CE2_CMLA_WRAP_ILLEGAL_KEK_PTR_ERROR
*   - CE2_CMLA_WRAP_ILLEGAL_KEY_PTR_ERROR
*   - CE2_CMLA_WRAP_ILLEGAL_WRAPPED_KEY_PTR_ERROR
*
* \brief \b 
* Description:
*  The CE2_CMLA_Wrap primitive composing CE2 AES Key Wrap Algorithm
*  and CMLA_DDT_Perm according to CMLA algorithm [CMLA v1.0-051221].
*
*  \b 
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function LLF_CMLA_Wrap. 
***************************************************************/
CE2CIMPORT_C  CE2Error_t CE2_CMLA_Wrap(CE2_CMLA_KEK_t            KEK,    /*in*/
                                    CE2_CMLA_UNWRAPPED_KEY_t  Key,    /*in*/
                                    CE2_CMLA_WRAPPED_KEY_t    WrapKey /*out*/ )
{
  if (KEK == DX_NULL)
    return CE2_CMLA_WRAP_ILLEGAL_KEK_PTR_ERROR;

  if (Key == DX_NULL)
    return CE2_CMLA_WRAP_ILLEGAL_KEY_PTR_ERROR;

  if (WrapKey == DX_NULL)
    return CE2_CMLA_WRAP_ILLEGAL_WRAPPED_KEY_PTR_ERROR;

  return LLF_CMLA_Wrap(KEK, Key, WrapKey);
} /* End of CE2_CMLA_Wrap */

/**
****************************************************************
* Function Name: 
*  CE2_CMLA_Unwrap
*
*  @param[in]  KEK      - Key Encryption Key, an octet string of length 16 bytes.
*  @param[in]  WrapKey  - Wrapped Key, an octet string of length 40 bytes.
*  @param[out] Key      - Plain key, an octet string of length 32 bytes.
*  
* @returns \b
*  CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - error code:
*   - CE2_CMLA_IS_NOT_SUPPORTED
*   - CE2_CMLA_UNWRAP_ILLEGAL_KEK_PTR_ERROR
*   - CE2_CMLA_UNWRAP_ILLEGAL_WRAPPED_KEY_PTR_ERROR
*   - CE2_CMLA_UNWRAP_ILLEGAL_KEY_PTR_ERROR
*
* \brief \b 
* Description:
*  The CE2_CMLA_Unwrap primitive performs inverse CE2_CMLA_Wrap 
*  transformation [CMLA v1.0-051221].
*
*  \b 
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function LLF_CMLA_Unwrap. 
***************************************************************/
CE2CIMPORT_C  CE2Error_t CE2_CMLA_Unwrap(CE2_CMLA_KEK_t            KEK,      /*in*/
                                      CE2_CMLA_WRAPPED_KEY_t    WrapKey,  /*in*/
                                      CE2_CMLA_UNWRAPPED_KEY_t  Key       /*out*/ )
{
  if (KEK == DX_NULL)
    return CE2_CMLA_UNWRAP_ILLEGAL_KEK_PTR_ERROR;

  if (Key == DX_NULL)
    return CE2_CMLA_UNWRAP_ILLEGAL_KEY_PTR_ERROR;

  if (WrapKey == DX_NULL)
    return CE2_CMLA_UNWRAP_ILLEGAL_WRAPPED_KEY_PTR_ERROR;

  return LLF_CMLA_Unwrap(KEK, WrapKey, Key);
} /* End of CE2_CMLA_Unwrap */

/**
****************************************************************
* Function Name: 
*  CE2_CMLA_RSA_Encrypt
*
*  @param[in]  PublKey     - A pointer to to structure containing user RSA Public Key.
*  @param[in]  Message	   - A pointer to message of length 128 bytes.
*  @param[out] EncrMessage - A pointer to output encrypted message of length 128 bytes.
*  
* @returns \b
*  CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - error code:
*   - CE2_CMLA_IS_NOT_SUPPORTED
*   - CE2_CMLA_RSA_ENCRYPT_ILLEGAL_PUB_KEY_PTR_ERROR
*   - CE2_CMLA_RSA_ENCRYPT_ILLEGAL_MESSAGE_PTR_ERROR
*   - CE2_CMLA_RSA_ENCRYPT_ILLEGAL_ENCR_MESSAGE_PTR_ERROR
*   - CE2_CMLA_RSA_ENCRYPT_MESSAGE_INTEGER_TOO_LONG_ERROR
*
* \brief \b 
* Description:
*  The CE2_CMLA_RSA_Encrypt primitive implements algorithm CMLA_RSA_Encrypt
*  [CMLA v1.0-051221].
*
*  \b 
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function LLF_CMLA_RSA_Encrypt. 
***************************************************************/
CE2CIMPORT_C  CE2Error_t  CE2_CMLA_RSA_Encrypt(
  CE2_RSAUserPubKey_t      *UserPublKey_ptr, /*in*/
  CE2_CMLA_RSA_MESSAGE_t    Message,         /*in*/
  CE2_CMLA_RSA_MESSAGE_t    EncrMessage     /*out*/)
{
  if (UserPublKey_ptr == DX_NULL)
    return CE2_CMLA_RSA_ENCRYPT_ILLEGAL_PUB_KEY_PTR_ERROR;

  if (Message == DX_NULL)
    return CE2_CMLA_RSA_ENCRYPT_ILLEGAL_MESSAGE_PTR_ERROR;

  if (EncrMessage == DX_NULL)
    return CE2_CMLA_RSA_ENCRYPT_ILLEGAL_ENCR_MESSAGE_PTR_ERROR;

  return LLF_CMLA_RSA_Encrypt(UserPublKey_ptr, Message, EncrMessage);
} /* End of CE2_CMLA_RSA_Encrypt */

/**
****************************************************************
* Function Name: 
*  CE2_CMLA_RSA_Decrypt
*
*  @param[in]  PrivKey     - A pointer to to structure containing user RSA Private Key.
*  @param[in]  EncrMessage - A pointer to input encrypted message of length 128 bytes.
*  @param[out] Message     - A pointer to output decrypted message of length 128 bytes.
*  
* @returns \b
*  CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - error code:
*   - CE2_CMLA_IS_NOT_SUPPORTED
*   - CE2_CMLA_RSA_ENCRYPT_ILLEGAL_PRIV_KEY_PTR_ERROR
*   - CE2_CMLA_RSA_ENCRYPT_ILLEGAL_MESSAGE_PTR_ERROR
*   - CE2_CMLA_RSA_ENCRYPT_ILLEGAL_ENCR_MESSAGE_PTR_ERROR
*   - CE2_CMLA_RSA_DECRYPT_MESSAGE_INTEGER_TOO_LONG_ERROR
*
* \brief \b 
* Description:
*  The CE2_CMLA_RSA_Decrypt primitive implements algorithm CMLA_RSA_Decrypt
*  [CMLA v1.0-051221].
*
*  \b 
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function LLF_CMLA_RSA_Decrypt. 
***************************************************************/
CE2CIMPORT_C  CE2Error_t  CE2_CMLA_RSA_Decrypt(
  CE2_RSAUserPrivKey_t     *UserPrivKey_ptr, /*in*/
  CE2_CMLA_RSA_MESSAGE_t    EncrMessage,     /*in*/
  CE2_CMLA_RSA_MESSAGE_t    Message         /*out*/)
{
  if (UserPrivKey_ptr == DX_NULL)
    return CE2_CMLA_RSA_DECRYPT_ILLEGAL_PRIV_KEY_PTR_ERROR;

  if (EncrMessage == DX_NULL)
    return CE2_CMLA_RSA_DECRYPT_ILLEGAL_ENCR_MESSAGE_PTR_ERROR;

  if (Message == DX_NULL)
    return CE2_CMLA_RSA_DECRYPT_ILLEGAL_MESSAGE_PTR_ERROR;

  return LLF_CMLA_RSA_Decrypt(UserPrivKey_ptr, EncrMessage, Message);
} /* End of CE2_CMLA_RSA_Decrypt */
